//----------------------------------------------------------------------------
// ObjectWindows
// Copyright (c) 1995, 1996 by Borland International, All Rights Reserved
//
//$Revision:   10.16  $
//
// Implementation of T31Button
//----------------------------------------------------------------------------
#include "main.h"
#pragma hdrstop
#if !defined(T31BTN_H)
# include "t31but.h"
#endif

const int  LayoutMargin         = 10;
const int  FaceToFocusRectDelta = -1;
const int  BUTTONSTATE_PUSHED   = 0x0004;
const int  BUTTONSTATE_FOCUS    = 0x0008;
const int  BUTTONSTYLE_MASK     = 0x00FF;
const long RopDSPDxax           = 0x00E20746L;

//
// Constructor of T31Button - Use this constructor to create a GlyphBtn
// from scratch.
//
T31Button::T31Button(TWindow* parent, int id, const char far* text,
                           int X, int Y, int W, int H, bool Left, bool isDefault,
                           TModule* module)
:
  TButton(parent, id, text, X, Y, W, H, isDefault, module)
{
  LeftSide=Left;
  InitVars();
}

//
// Method used to initialized variables used by GlyphButton's implementation
//
void
T31Button::InitVars()
{
  UpBmp = 0;
  DownBmp = 0;
  FocusBmp = 0;
  DisabledBmp = 0;
  BtnFont = new TDefaultGUIFont;
  xText = yText = -1;
  xGlyph = yGlyph = -1;
  LayStyle = lsH_GST;
  Set(biShowText);
}

//
// Destructor - Cleanup resources used by Glyph Button object
//
T31Button::~T31Button()
{
  delete UpBmp;
  delete DownBmp;
  delete FocusBmp;
  delete DisabledBmp;
  delete BtnFont;
}

//
// Response Table
//
DEFINE_RESPONSE_TABLE1(T31Button, TButton)
  EV_WM_PAINT,
  EV_WM_ERASEBKGND,
  EV_WM_SETFOCUS,
  EV_WM_KILLFOCUS,
  EV_WM_GETFONT,
  EV_WM_SETFONT,
  EV_WM_GETDLGCODE,
  EV_WM_LBUTTONDOWN,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_LBUTTONUP,
  EV_WM_MOUSEMOVE,
  EV_WM_KEYDOWN,
  EV_WM_KEYUP,
  EV_WM_ENABLE,
  EV_WM_CANCELMODE,
  EV_MESSAGE(BM_SETSTATE, BmSetState),
  EV_MESSAGE(BM_GETSTATE, BmGetState),
  EV_MESSAGE(BM_SETSTYLE, BMSetStyle),
END_RESPONSE_TABLE;

//
//
//
LRESULT CALLBACK OWL_EXPORT16
BButtonProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
  switch(msg) {
    case WM_GETDLGCODE: {
      uint32 style = GetWindowLong(hwnd, GWL_STYLE);
      uint16 btnStyle = uint16(LoUint16(style) & BUTTONSTYLE_MASK);
      if (btnStyle == BS_DEFPUSHBUTTON)
        return DLGC_BUTTON|DLGC_DEFPUSHBUTTON;
      else
        return DLGC_BUTTON|DLGC_UNDEFPUSHBUTTON;
    }

    case BM_SETSTATE: {
      static int i;
      i++;
      break;
    }

    case BM_GETSTATE: {
      static int i;
      i++;
      break;
    }

    case BM_SETSTYLE:   {
      static int i;
      i++;
      break;
    }
  }
  return DefWindowProc(hwnd, msg, wParam, lParam);
}

//
// Overriden virtual of TWindow - Fills out information about the Window
// class associated with a glyph button.
// NOTE: The class information is based on the system's "BUTTON" class.
//
void
T31Button::GetWindowClass(WNDCLASS& wndClass)
{
  // Grab a the attributes of the native "BUTTON" control
  //
  if (::GetClassInfo(0, "BUTTON", &wndClass)) {
    wndClass.hInstance = *GetModule();
    wndClass.lpszClassName = GetClassName();
    wndClass.lpfnWndProc = BButtonProc;
  }
  else {
    TControl::GetWindowClass(wndClass);
    wndClass.style = CS_HREDRAW|CS_VREDRAW|CS_PARENTDC;
    wndClass.hbrBackground = HBRUSH(COLOR_BTNFACE+1);
  }
}

void	T31Button::SetCaption(const char far* title)
{
	TButton::SetCaption(title);
   Invalidate();
}

//
// Return name of window class associated with a glyph button control.
//
char far*
T31Button::GetClassName()
{
  return OWL_GLYPHBTN;
}

//
// Overriden virtual of TWindow - Updates internal flags based on style of
// underlying window.
//
void
T31Button::SetupWindow()
{
  TButton::SetupWindow();

  // Update state flags based on current style
  //
  uint32 style = GetStyle();
  if (style & BS_DEFPUSHBUTTON)
    Set(biDefault);
  if (style & WS_DISABLED)
    Set(biDisabled);
}

//
// Specifies a bitmap to be used as glyph
//
void
T31Button::SetGlyph(HBITMAP hBitmap, TGlyphType type, TAutoDelete autoDelete)
{
#if defined(BI_PLAT_WIN16)
  PRECONDITION(!hBitmap || ::IsGDIObject(hBitmap) != 0);
#else
  PRECONDITION(!hBitmap || ::GetObjectType(hBitmap) == OBJ_BITMAP);
#endif

  SetGlyph(hBitmap ? new TBitmap(hBitmap, autoDelete) : (TBitmap*)0, type);
}

//
// Specify the resource identifier of a bitmap to be used as glyph
//
void
T31Button::SetGlyph(TResId resId, TModule* module, TGlyphType type)
{
  PRECONDITION(module != 0 || GetModule());
  SetGlyph(new TBitmap(module ? *module : *GetModule(), resId), type);
}

//
// Specify a bitmap object to be used as glyph
// NOTE: The 'bitmap' parameter can be 0 to reset the glyph stored by the
//       glyph button object.
//
void
T31Button::SetGlyph(TBitmap* bitmap, TGlyphType type)
{
  PRECONDITION(!bitmap || bitmap->IsGDIObject());

  switch (type) {
    case gtUp:
      delete UpBmp;
      UpBmp = bitmap;
      break;

    case gtDown:
      delete DownBmp;
      DownBmp = bitmap;
      break;

    case gtFocus:
      delete FocusBmp;
      FocusBmp = bitmap;
      break;

    case gtDisabled:
      delete DisabledBmp;
      DisabledBmp = bitmap;
      break;

    default:
      break;
  }

  // Update status flag
  //
  if (UpBmp)
    Set(biShowGlyph);
  else
    Clear(biShowGlyph);
}

//
// WM_PAINT handler - Invokes 'Paint' method to display glyph and/or text.
//
void
T31Button::EvPaint()
{
  TPaintDC dc(*this);
  TRect&   rect = *(TRect*)&dc.Ps.rcPaint;
  Paint(dc, dc.Ps.fErase, rect);
}

//
// Invoke 'PaintButton' to display glyph and/or text.
//
void
T31Button::Paint(TDC& dc, bool /*erase*/, TRect& /*rect*/)
{
  PaintButton(dc);
}


//
// WM_ERASEBKGND handler - Return true to prevent background from being
// erased. (WM_PAINT handler paints whole client area).
//
bool
T31Button::EvEraseBkgnd(HDC /*dc*/)
{
  return true;
}

//
// WM_SETFOCUS handler - Updates internal flag and forces button to redraw
//
void
T31Button::EvSetFocus(THandle /*hWndLostFocus*/)
{
  Set(biFocus);
  Invalidate(true);
}

//
// WM_KILLFOCUS handler - Updates internal flag and forces button to redraw
//
void
T31Button::EvKillFocus(THandle /*hWndGetFocus*/)
{
  Clear(biFocus);
  if (IsSet(biPushed))
    ClearCapture();
  else
    Invalidate(true);
}

//
// WM_GETFONT handler - Returns font used by control if one was specified
// earlier. Otherwise, returns 0.
//
HFONT
T31Button::EvGetFont()
{
  PRECONDITION(!BtnFont || BtnFont->IsGDIObject());
  return BtnFont ? HFONT(*BtnFont) : HFONT(0);
}

//
// WM_SETFONT Handler. Deletes any cached font and stores copy of new one.
//
void
T31Button::EvSetFont(HFONT hFont, bool redraw)
{
  delete BtnFont;
  BtnFont = new TFont(hFont);
  if (redraw)
    Invalidate();
}

//
// WM_GETDLGCODE handler. Inform dialog manager that we're a 'normal' push
// button or the default push button according to our style.
//
uint
T31Button::EvGetDlgCode(MSG far* /*msg*/)
{
  if (IsSet(biDefault))
    return DLGC_BUTTON|DLGC_DEFPUSHBUTTON;
  else
    return DLGC_BUTTON|DLGC_UNDEFPUSHBUTTON;
}

//
// WM_LBUTTONDOWN handler. Grab focus and update button's state to be in
// 'pushed' mode.
//
void
T31Button::EvLButtonDown(uint /*modKeys*/, TPoint& /*point*/)
{
  SetCapture();
  SendMessage(BM_SETSTATE, TRUE);
  if (!IsSet(biFocus))
    SetFocus();
}

//
// WM_LBUTTONDBLCLK handler. Simply forward to LBUTTONDOWN handler
//
void
T31Button::EvLButtonDblClk(uint modKeys, TPoint& point)
{
  EvLButtonDown(modKeys, point);
}

//
// WM_LBUTTONUP handler. Restore state of button and notify parent with a
// CLICKED message if necessary.
//
void
T31Button::EvLButtonUp(uint /*modKeys*/, TPoint& point)
{
  if (GetCapture() == *this) {
    ReleaseCapture();

    SendMessage(BM_SETSTATE, FALSE);

    TRect rect;
    GetClientRect(rect);

    if (rect.Contains(point)) {
      SendNotification(::GetParent(*this), GetDlgCtrlID(),
                       BN_CLICKED, *this);
    }
  }
}

//
// WM_MOUSEMOVE handler. Update state of button if we're in 'capture' mode.
//
void
T31Button::EvMouseMove(uint modKeys, TPoint& point)
{
  if (modKeys & MK_LBUTTON && GetCapture() == *this) {
    TRect rect;
    GetClientRect(rect);

    if (rect.Contains(point))
      SendMessage(BM_SETSTATE, TRUE);
    else
      SendMessage(BM_SETSTATE, FALSE);
  }
}

//
// WM_KEYDOWN handler. Update state of button upon detecting that user
// pressed the space bar.
//
void
T31Button::EvKeyDown(uint key, uint /*repeatCount*/, uint /*flags*/)
{
  if (key == VK_SPACE)
    SendMessage(BM_SETSTATE, TRUE);
}

//
// WM_KEYUP handler. Restore state of button and notify parent
//
void
T31Button::EvKeyUp(uint key, uint /*repeatCount*/, uint /*flags*/)
{
  if (IsSet(biPushed) && key == VK_SPACE) {
    SendMessage(BM_SETSTATE, FALSE);

    SendNotification(::GetParent(*this), GetDlgCtrlID(),
                      BN_CLICKED, *this);
  }
}

//
// WM_ENABLE handler. Update internal flags and invalidate control if
// necessary.
//
void
T31Button::EvEnable(bool enabled)
{
  if (enabled) {
    Clear(biDisabled);
  }
  else {
    ClearCapture();
    Set(biDisabled);
  }
  Invalidate(true);
}

//
//
//
void
T31Button::EvCancelMode()
{
  ClearCapture();
}

//
// BM_GETSTATE handler. Return the current state of the window.
//
TResult
T31Button::BmGetState(TParam1 /*param1*/, TParam2 /*param2*/)
{
  TResult result = 0;
  if (IsSet(biPushed))
    result |= BUTTONSTATE_PUSHED;

  if (IsSet(biFocus))
    result |= BUTTONSTATE_FOCUS;

  return result;
}

//
// BM_SETSTATE handler. Update internal state flags based on parameters and
// redraw control if necessary.
//
TResult
T31Button::BmSetState(TParam1 param1, TParam2 /*param2*/)
{
  if (param1) {
    // Needs hilight look
    //
    if (!IsSet(biPushed)) {
      Set(biPushed);
      PaintNow();
    }
  }
  else {
    // Needs normal look
    //
    if (IsSet(biPushed)) {
      Clear(biPushed);
      PaintNow();
    }
  }
  return 0;
}

//
// BM_SETSTYLE handler. Update internal flags to match specified parameters
// and invalidate the window if necessary.
//
TResult
T31Button::BmSetStyle(TParam1 param1, TParam2 /*param2*/)
{
  // Grab and splice the styles
  //
  uint32 style = GetStyle();
  uint16 winStyle = HiUint16(style);
  uint16 btnStyle = uint16(LoUint16(style) & BUTTONSTYLE_MASK);

  // Check against passed in parameter
  // NOTE: We only cater to PUSHBUTTON and DEFPUSHBUTTON
  //       The current definition so BS_PUSHBUTTON is 0L
  //
  if (LOWORD(param1) == BS_PUSHBUTTON  &&  btnStyle != BS_PUSHBUTTON) {

    // Make 'simple' push button
    //
    TWindow::SetStyle(MAKELONG(param1, winStyle));
    Clear(biDefault);
    Invalidate(true);
  }
  else if (LOWORD(param1)==BS_DEFPUSHBUTTON && btnStyle != BS_DEFPUSHBUTTON){

    // Make 'default' push button
    //
    TWindow::SetStyle(MAKELONG(param1, winStyle));
    Set(biDefault);
    Invalidate(true);
  }

#if defined(__DEBUG) || defined(__TRACE) || defined(__WARN)
  if (LOWORD(param1) != BS_PUSHBUTTON &&
      LOWORD(param1) != BS_DEFPUSHBUTTON)
    TRACEX(OwlControl, 0, "BmSetStyle: Invalid style specified");
#endif

  return 0;
}

//
// Release caption if we are in 'capture' mode. Reset internal flags
// appropriately.
//
void
T31Button::ClearCapture()
{
  if (GetCapture() == *this)
    ReleaseCapture();
  Clear(biPushed);
  Invalidate(true);
}

//
// Paint the button into a memory DC and bitblt the final rendering to the
// specified 'dc'.
//
void
T31Button::PaintButton(TDC& dc)
{
  TRect rect;
  GetClientRect(rect);

  // Create compatible bitmap
  //
  TBitmap memBmp(dc, rect.Width(), rect.Height());

  // Create compatible memory DC
  //
  TMemoryDC memDC(dc);

  // Select and init memory bitmap
  //
  memDC.SelectObject(memBmp);

  // Save the rectangle for bitblt'ing - 'rect' will be adjusted
  // as we move from defaultFrame, border and face...
  //
  TRect bltRect = rect;

  // Paint the button into the memory DC
  //
  PaintFrame(memDC, rect);
  PaintFace(memDC, rect);

  // Bitblt the button to the output device
  //
  dc.BitBlt(bltRect, memDC, TPoint(bltRect.left, bltRect.top));

  // Cleanup
  //
  memDC.RestoreBitmap();
}

//
// Draw the border of the button
//
void
T31Button::PaintFrame(TDC& dc, TRect& rect)
{
   int x;
  	rect = TRect(rect.left, rect.top, rect.right-1, rect.bottom-1);
	TPen PenBoven((IsSet(biPushed) ? (TColor::Black) : (TColor(255,255,255))));
	TPen PenOnder((IsSet(biPushed) ? (TColor::Sys3dLight) : (TColor::Black)));
   TColor ColorFace((IsSet(biPushed) ? (TColor::LtYellow) : (TColor::Sys3dFace)));
   TPen PenFace(ColorFace);
   TBrush BrushFace(ColorFace);
   TBrush Back(TColor(0,128,0));

   //bepalen waar de cirkel moet komen
   if (LeftSide) x=0; else x=rect.right-20;

   dc.FillRect(x, rect.top, x+21, rect.bottom+1, Back);

	// ButtonFace tekenen
   dc.SelectObject(PenFace);
   dc.SelectObject(BrushFace);
	dc.Ellipse(x, rect.top, x+20, rect.bottom);
   dc.ExtFloodFill(TPoint(x+10, rect.top+10), ColorFace, FLOODFILLBORDER);

   if (LeftSide)
   	{
		dc.FillRect(x+10, rect.top, rect.right, rect.bottom, BrushFace);
      }
   else
   	{
		dc.FillRect(x+10, rect.top, rect.left, rect.bottom, BrushFace);
		}

   // Teken de outline
	dc.SelectObject(PenOnder);
	if (LeftSide)
   	{
		dc.Arc(x, rect.top, x+20, rect.bottom, x, rect.bottom, x+10, rect.bottom);
   	dc.MoveTo(x+10, rect.bottom);
   	dc.LineTo(rect.right, rect.bottom);
      dc.LineTo(rect.right, rect.top);
		}
   else
   	{
      dc.Arc(x,rect.top, x+20, rect.bottom, x+10, rect.bottom, x+20, rect.top);
      dc.MoveTo(x+10, rect.bottom);
      dc.LineTo(rect.left, rect.bottom);
		}
   dc.SelectObject(PenBoven);
	if (LeftSide)
   	{
		dc.Arc(x, rect.top, x+20, rect.bottom, x+10, rect.top, x, rect.bottom);
   	dc.MoveTo(x+10, rect.top);
   	dc.LineTo(rect.right, rect.top);
		}
   else
   	{
      dc.Arc(x,rect.top, x+20, rect.bottom, x+20, rect.top, x+10, rect.top);
      dc.MoveTo(x+10, rect.top);
      dc.LineTo(rect.left, rect.top);
      dc.LineTo(rect.left, rect.bottom);
      }

  // Shrink the rectangle to leave the face
  rect = TRect(rect.left+2, rect.top+2, rect.right-2, rect.bottom-2);
}

//
// Draw the face of the button [i.e. text and glyph portions]
//
void
T31Button::PaintFace(TDC& dc, TRect& rect)
{
  	// Fill the background with the face color
  	//
  TRect rect2(rect.left+11, rect.top, rect.right-11, rect.bottom);
  	if (IsSet(biPushed))
   	{
      TBrush brush(TColor::LtYellow);
  		dc.FillRect(rect2, brush);
      }
   else
   	{
      TBrush brush(TColor::Sys3dFace);
  		dc.FillRect(rect2, brush);
  		}


  // Grab the glyph and it's size
  //
  TBitmap* glyph = 0;             // Pointer to glyph
  TRect glyphRect(0, 0, 0, 0);    // Size of glyph
  if (IsSet(biShowGlyph)) {
    // Start with the up bitmap
    //
    glyph = UpBmp;

    // Switch to more appropriate bitmap if applicable
    //
    if (IsSet(biPushed))
    	{
      if (DownBmp)
      	{
      	glyph = DownBmp;
         }
      } 
    else if (IsSet(biDisabled) && DisabledBmp)
      glyph = DisabledBmp;

    CHECK(glyph && glyph->IsGDIObject());
    glyphRect.Set(0, 0, glyph->Width(), glyph->Height());
  }

  // Grab some information about the text/caption
  //
  int len = 0;                    // Length of Caption
  TRect textRect(0, 0, 0, 0);     // Size of text
  TAPointer<char> text;           // Pointer to caption dynamic buffer
  TPointer<TFont> tmpFnt;         // Object wrapping font handle
  if (IsSet(biShowText)) {
    len = GetWindowTextLength();
    if (len) {

      // Select the font
      //
      if (!BtnFont) {
        HFONT hFont = HFONT(::SendMessage(::GetParent(*this), WM_GETFONT,
                                          0, 0));
        if (!hFont)
          hFont = HFONT(GetStockObject(TSystem::Has3dUI() ? ANSI_VAR_FONT :
                                                            SYSTEM_FONT));
        if (hFont)
          tmpFnt = new TFont(hFont);
      }
      if (BtnFont) {
        CHECK(BtnFont->IsGDIObject());
        dc.SelectObject(*BtnFont);
      }
      else if (tmpFnt) {
        CHECK(((TFont&)tmpFnt).IsGDIObject());
        dc.SelectObject((TFont&)tmpFnt);
      }

      text = new char[len+1];
      GetWindowText(text, len+1);
      textRect.Set(0, 0, rect.Width() - glyphRect.Width(), SHRT_MAX);

      // Display text in proper color
      //
      TColor textColor;
      if (IsSet(biDisabled)) {
        if (TSystem::IsWin95())
          textColor = TColor::SysGrayText;
        else
          textColor = TColor::Gray;
      }
      else
        textColor = TColor::SysBtnText;

      TColor oldColor = dc.SetTextColor(textColor);
      TFont ButtonFont("MS Sans Serif", 14, 0, 0, 0, FW_BOLD);
      dc.SelectObject(ButtonFont);

      dc.DrawText(text, len, textRect, DT_WORDBREAK|DT_CALCRECT);
      dc.SetTextColor(oldColor);
    }
  }

  // If we have text and/or glyph, lay them out and paint them
  //
  if (!textRect.IsNull() || !glyphRect.IsNull()) {

    LayoutTextGlyph(rect, textRect, glyphRect);

    // Offset things to the lower right if we're in down
    //
    if (IsSet(biPushed)) {
        glyphRect.Offset(1, 1);
      if (!glyphRect.IsNull() && glyph == UpBmp)
        glyphRect.Offset(1, 1);
      if (!textRect.IsNull())
        textRect.Offset(1, 1);
    }

    // Paint the components of the button
    //
    if (!glyphRect.IsNull()) {
      PRECONDITION(glyph && glyph->IsGDIObject());
      TUIFace uiFace(glyphRect, *glyph);
      uiFace.Paint(dc, TPoint(0, 0), TUIFace::Normal, false, true);
    }
    if (!textRect.IsNull()) {
      int mode = dc.SetBkMode(TRANSPARENT);
      TColor oldColor = dc.SetTextColor(IsSet(biDisabled) ? TColor::SysGrayText :
                                                            TColor::SysBtnText);
      TFont ButtonFont("MS Sans Serif", 14, 0, 0, 0, FW_BOLD);
      dc.SelectObject(ButtonFont);
      dc.DrawText(text, len, textRect, DT_WORDBREAK);
      dc.SetTextColor(oldColor);
      dc.SetBkMode(mode);
    }
  }

  // Restore font
  //
  if (len && (BtnFont || tmpFnt))
    dc.RestoreFont();
}

//
// Virtual routine invoked to retrieve the placement of text and glyph
// when drawing the button.
//
void
T31Button::LayoutTextGlyph(const TRect& faceRect, TRect& textRect,
                              TRect& glyphRect)
{
  // Must have either text or a glyph
  //
  PRECONDITION(!textRect.IsNull() || !glyphRect.IsNull());

  // First check for the case where we've got either
  // text or glyph - but not both
  //
  if (textRect.IsNull() || glyphRect.IsNull()) {
    TRect& centerRect = textRect.IsNull() ? glyphRect : textRect;

    centerRect.Offset(faceRect.left, faceRect.top);
    if (centerRect.Width() < faceRect.Width())
      centerRect.Offset((faceRect.Width() - centerRect.Width())/2, 0);
    else
      centerRect.right = faceRect.right;
    if (centerRect.Height() < faceRect.Height())
      centerRect.Offset(0, (faceRect.Height() - centerRect.Height())/2);
    else
      centerRect.bottom = faceRect.bottom;
  }

  // Here we attempt to layout both the glyph and text
  //
  else {

    // Align upper left corners of face, text and glyph rectangles
    //
    glyphRect.Offset(faceRect.left, faceRect.top);
    textRect.Offset(faceRect.left, faceRect.top);

    // Compute amount of 'extra' space, if any, and how to partition it
    // between the two items
    //
    int space  = faceRect.Width() - glyphRect.Width()- textRect.Width() -
                 LayoutMargin*2;
    int gDelta;
    int tDelta;

    switch (LayStyle) {
     case lsH_SGST: {
              gDelta = space >= 0 ? LayoutMargin + space/3 :
                                    LayoutMargin + space/2;
              tDelta = space >= 0 ? gDelta + glyphRect.Width() + space/3 :
                                    gDelta + glyphRect.Width();
            }
            break;

      case lsH_GST: {
              gDelta = space >= 0 ? LayoutMargin : LayoutMargin + space/2;
              tDelta = space >= 0 ? gDelta + glyphRect.Width() + space/2 :
                                    gDelta + glyphRect.Width();
            }
            break;

      case lsH_STSG:
      case lsH_TSGS:
      case lsV_SGST:
      case lsV_GST:
      case lsV_STSG:
      case lsV_TSGS:
            break;

      default:
        break;
    }

    if (LayStyle == lsH_SGST || LayStyle == lsH_GST ||
        LayStyle == lsH_STSG || LayStyle == lsH_TSGS ) {

      // Center vertically
      //
      if (textRect.Height() < faceRect.Height())
        textRect.Offset(0, (faceRect.Height() - textRect.Height())/2);
      if (glyphRect.Height() < faceRect.Height())
        glyphRect.Offset(0, (faceRect.Height() - glyphRect.Height())/2);

      // Layout horizontally
      //
      glyphRect.Offset(gDelta, 0);
      textRect.Offset(tDelta, 0);
    }
    else if (LayStyle == lsV_SGST || LayStyle == lsV_GST ||
             LayStyle == lsV_STSG || LayStyle == lsV_TSGS ) {

      // Or center horizontally
      //
      if (textRect.Width() < faceRect.Width())
        textRect.Offset((faceRect.Width() - textRect.Width())/2, 0);
      if (glyphRect.Width() < faceRect.Width())
        glyphRect.Offset((faceRect.Width() - glyphRect.Width())/2, 0);

      // Layout vertically
      //
      glyphRect.Offset(0, gDelta);
      textRect.Offset(0, tDelta);
    }
  }
}

//
// Specify a 'style' describing how text and glyph should be laid out.
// Invalidate the window if necessary.
//
void
T31Button::SetLayoutStyle(TLayoutStyle style)
{
  if (style != LayStyle) {
    LayStyle = style;
    if (GetHandle())
      Invalidate();
  }
}

//
// Sets text coordinates and invalidates window if necessary
//
void
T31Button::SetTextOrigin(int x, int y)
{
  if (x != xText || y != yText) {
    xText = x;
    yText = y;
    if (GetHandle())
      Invalidate();
  }
}

//
// Set the upper left corner of glyphs and invalidates window if necessary
//
void
T31Button::SetGlyphOrigin(int x, int y)
{
  if (x != xGlyph || y != yGlyph) {
    xGlyph = x;
    yGlyph = y;
    if (GetHandle())
      Invalidate();
  }
}

//
// Display a focus rectangle
//
void
T31Button::PaintFocusRect(TDC& dc, const TRect& faceRect)
{
  PRECONDITION(IsSet(biFocus));
  TRect focusRect = faceRect;
  focusRect.Inflate(FaceToFocusRectDelta, FaceToFocusRectDelta);
  dc.DrawFocusRect(focusRect);
}

//
// Repaints window right away by retrieving a client DC and invoking the
// 'Paint' method.
//
void
T31Button::PaintNow()
{
  TRect rect;
  GetClientRect(rect);
  TClientDC dc(*this);
  Paint(dc, false, rect);
}
